<?php
/**
 * ConnectWise Sync API – Register, Login, Get Sync Token
 * - register | login: { "action", "email", "password" } – legacy; user auth is via browsx.
 * - get_sync_token: { "action": "get_sync_token", "email": "...", "license_key": "..." } – no password; used after browsx validated user.
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  http_response_code(204);
  exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
  exit;
}

require_once __DIR__ . '/../config.php';

$raw = file_get_contents('php://input');
$input = json_decode($raw, true);
if (!is_array($input)) {
  echo json_encode(['ok' => false, 'error' => 'Invalid JSON']);
  exit;
}

$action = isset($input['action']) ? trim($input['action']) : '';
$email = isset($input['email']) ? trim(strtolower($input['email'])) : '';
$password = isset($input['password']) ? $input['password'] : '';
$fullName = isset($input['full_name']) ? trim($input['full_name']) : '';
$licenseKey = isset($input['license_key']) ? trim($input['license_key']) : '';

// get_sync_token: no password; email + license_key only (call after browsx validated user)
if ($action === 'get_sync_token') {
  if (!strlen($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['ok' => false, 'error' => 'Valid email required']);
    exit;
  }
  if (defined('USE_MYSQL') && USE_MYSQL && function_exists('friender_db')) {
    try {
      $pdo = friender_db();
      $stmt = $pdo->prepare('SELECT id FROM users WHERE email = ?');
      $stmt->execute([$email]);
      $user = $stmt->fetch();
      $now = time();
      $expires = $now + (365 * 24 * 3600);
      $token = bin2hex(random_bytes(24));
      if ($user) {
        $userId = (int) $user['id'];
      } else {
        $internalPassword = bin2hex(random_bytes(16));
        $hash = password_hash($internalPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare('INSERT INTO users (email, password_hash, full_name, created_at) VALUES (?, ?, NULL, ?)');
        $stmt->execute([$email, $hash, $now]);
        $userId = (int) $pdo->lastInsertId();
      }
      $stmt = $pdo->prepare('INSERT INTO tokens (user_id, token, expires_at, created_at) VALUES (?, ?, ?, ?)');
      $stmt->execute([$userId, $token, $expires, $now]);
      echo json_encode(['ok' => true, 'token' => $token, 'email' => $email]);
      exit;
    } catch (PDOException $e) {
      error_log('ConnectWise get_sync_token error: ' . $e->getMessage());
      echo json_encode(['ok' => false, 'error' => 'Server error']);
      exit;
    }
  }
  echo json_encode(['ok' => false, 'error' => 'Server not configured']);
  exit;
}

if (!strlen($email) || !strlen($password)) {
  echo json_encode(['ok' => false, 'error' => 'Email and password required']);
  exit;
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
  echo json_encode(['ok' => false, 'error' => 'Invalid email']);
  exit;
}

if (defined('USE_MYSQL') && USE_MYSQL && function_exists('friender_db')) {
  // --- MySQL ---
  try {
    $pdo = friender_db();
    if ($action === 'register') {
      $stmt = $pdo->prepare('SELECT id FROM users WHERE email = ?');
      $stmt->execute([$email]);
      if ($stmt->fetch()) {
        echo json_encode(['ok' => false, 'error' => 'Email already registered']);
        exit;
      }
      $hash = password_hash($password, PASSWORD_DEFAULT);
      $now = time();
      $stmt = $pdo->prepare('INSERT INTO users (email, password_hash, full_name, created_at) VALUES (?, ?, ?, ?)');
      $stmt->execute([$email, $hash, $fullName ?: null, $now]);
      $userId = (int) $pdo->lastInsertId();
      $token = bin2hex(random_bytes(24));
      $expires = $now + (365 * 24 * 3600);
      $stmt = $pdo->prepare('INSERT INTO tokens (user_id, token, expires_at, created_at) VALUES (?, ?, ?, ?)');
      $stmt->execute([$userId, $token, $expires, $now]);
      echo json_encode(['ok' => true, 'token' => $token, 'email' => $email]);
      exit;
    }
    if ($action === 'login') {
      $stmt = $pdo->prepare('SELECT id, password_hash FROM users WHERE email = ?');
      $stmt->execute([$email]);
      $user = $stmt->fetch();
      if (!$user || !password_verify($password, $user['password_hash'])) {
        echo json_encode(['ok' => false, 'error' => 'Invalid email or password']);
        exit;
      }
      $userId = (int) $user['id'];
      $token = bin2hex(random_bytes(24));
      $now = time();
      $expires = $now + (365 * 24 * 3600);
      $stmt = $pdo->prepare('INSERT INTO tokens (user_id, token, expires_at, created_at) VALUES (?, ?, ?, ?)');
      $stmt->execute([$userId, $token, $expires, $now]);
      echo json_encode(['ok' => true, 'token' => $token, 'email' => $email]);
      exit;
    }
  } catch (PDOException $e) {
    error_log('ConnectWise auth DB error: ' . $e->getMessage());
    echo json_encode(['ok' => false, 'error' => 'Server error']);
    exit;
  }
  echo json_encode(['ok' => false, 'error' => 'Unknown action. Use register or login']);
  exit;
}

// --- File-based (fallback) ---
$users = [];
if (file_exists(USERS_FILE)) {
  $users = json_decode(file_get_contents(USERS_FILE), true);
  if (!is_array($users)) $users = [];
}

$userId = 'u_' . md5($email . 'connectwise_salt_2025');

if ($action === 'register') {
  if (isset($users[$userId])) {
    echo json_encode(['ok' => false, 'error' => 'Email already registered']);
    exit;
  }
  $users[$userId] = [
    'email' => $email,
    'hash'  => password_hash($password, PASSWORD_DEFAULT),
    'created' => time(),
  ];
  if (!file_put_contents(USERS_FILE, json_encode($users, JSON_PRETTY_PRINT))) {
    echo json_encode(['ok' => false, 'error' => 'Registration failed']);
    exit;
  }
  $token = bin2hex(random_bytes(24));
  $tokenFile = STORAGE_DIR . '/tokens/' . $userId . '.txt';
  $tokenDir = dirname($tokenFile);
  if (!is_dir($tokenDir)) mkdir($tokenDir, 0755, true);
  file_put_contents($tokenFile, $token . "\n" . (time() + 365 * 24 * 3600));
  echo json_encode(['ok' => true, 'token' => $token, 'email' => $email]);
  exit;
}

if ($action === 'login') {
  if (!isset($users[$userId]) || !password_verify($password, $users[$userId]['hash'])) {
    echo json_encode(['ok' => false, 'error' => 'Invalid email or password']);
    exit;
  }
  $token = bin2hex(random_bytes(24));
  $tokenFile = STORAGE_DIR . '/tokens/' . $userId . '.txt';
  $tokenDir = dirname($tokenFile);
  if (!is_dir($tokenDir)) mkdir($tokenDir, 0755, true);
  file_put_contents($tokenFile, $token . "\n" . (time() + 365 * 24 * 3600));
  echo json_encode(['ok' => true, 'token' => $token, 'email' => $email]);
  exit;
}

echo json_encode(['ok' => false, 'error' => 'Unknown action. Use register or login']);
exit;
