<?php
/**
 * ConnectWise Sync API – Load data from server
 * GET with Header: Authorization: Bearer <token>
 * Returns full sync payload (all keys to restore in extension).
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  http_response_code(204);
  exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
  http_response_code(405);
  echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
  exit;
}

require_once __DIR__ . '/../config.php';

$auth = isset($_SERVER['HTTP_AUTHORIZATION']) ? $_SERVER['HTTP_AUTHORIZATION'] : '';
if (!preg_match('/^Bearer\s+(.+)$/i', $auth, $m)) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Missing or invalid Authorization']);
  exit;
}
$token = trim($m[1]);

$userId = null;
if (defined('USE_MYSQL') && USE_MYSQL && function_exists('friender_resolve_token')) {
  $userId = friender_resolve_token($token);
}
if ($userId === null) {
  $userId = resolveTokenToUserIdFile($token);
}

if ($userId === null) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Invalid or expired token']);
  exit;
}

if (defined('USE_MYSQL') && USE_MYSQL && function_exists('friender_db') && is_int($userId)) {
  try {
    $pdo = friender_db();
    $stmt = $pdo->prepare('SELECT payload, synced_at FROM sync_data WHERE user_id = ?');
    $stmt->execute([$userId]);
    $row = $stmt->fetch();
    if (!$row || !$row['payload']) {
      echo json_encode(['ok' => true, 'data' => null, 'message' => 'No saved data yet']);
      exit;
    }
    $payload = json_decode($row['payload'], true);
    if (!is_array($payload)) {
      echo json_encode(['ok' => false, 'error' => 'Corrupt data']);
      exit;
    }
    unset($payload['_syncedAt']);
    echo json_encode(['ok' => true, 'data' => $payload]);
    exit;
  } catch (PDOException $e) {
    error_log('ConnectWise load DB error: ' . $e->getMessage());
    echo json_encode(['ok' => false, 'error' => 'Server error']);
    exit;
  }
}

// File-based
$dataFile = STORAGE_DIR . '/sync/' . $userId . '.json';
if (!file_exists($dataFile)) {
  echo json_encode(['ok' => true, 'data' => null, 'message' => 'No saved data yet']);
  exit;
}
$payload = json_decode(file_get_contents($dataFile), true);
if (!is_array($payload)) {
  echo json_encode(['ok' => false, 'error' => 'Corrupt data']);
  exit;
}
unset($payload['_syncedAt']);
echo json_encode(['ok' => true, 'data' => $payload]);
exit;

function resolveTokenToUserIdFile($token) {
  if (!strlen($token)) return null;
  $tokenDir = STORAGE_DIR . '/tokens';
  if (!is_dir($tokenDir)) return null;
  foreach (glob($tokenDir . '/*.txt') as $f) {
    $content = file_get_contents($f);
    $lines = explode("\n", $content);
    $stored = trim($lines[0] ?? '');
    $expires = (int)($lines[1] ?? 0);
    if ($stored === $token && $expires > time()) {
      return basename($f, '.txt');
    }
  }
  return null;
}
