<?php
/**
 * ConnectWise Sync API – Save data to server
 * POST JSON body = full sync payload. Header: Authorization: Bearer <token>
 * Writes to: sync_config, sync_segments, sync_groups, sync_automation_state,
 *   sync_activity_log, sync_pending_friend_requests, sync_position_resume,
 *   sync_pending_message_retry, sync_tracking_sets, sync_misc, sync_data
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  http_response_code(204);
  exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
  exit;
}

require_once __DIR__ . '/../config.php';

$auth = isset($_SERVER['HTTP_AUTHORIZATION']) ? $_SERVER['HTTP_AUTHORIZATION'] : '';
if (!preg_match('/^Bearer\s+(.+)$/i', $auth, $m)) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Missing or invalid Authorization']);
  exit;
}
$token = trim($m[1]);

$userId = null;
if (defined('USE_MYSQL') && USE_MYSQL && function_exists('friender_resolve_token')) {
  $userId = friender_resolve_token($token);
}
if ($userId === null) {
  $userId = resolveTokenToUserIdFile($token);
}

if ($userId === null) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Invalid or expired token']);
  exit;
}

$raw = file_get_contents('php://input');
if (strlen($raw) > MAX_PAYLOAD_BYTES) {
  http_response_code(413);
  echo json_encode(['ok' => false, 'error' => 'Payload too large']);
  exit;
}

$payload = json_decode($raw, true);
if (!is_array($payload)) {
  echo json_encode(['ok' => false, 'error' => 'Invalid JSON']);
  exit;
}

$payload['_syncedAt'] = time();
$syncedAt = $payload['_syncedAt'];

if (defined('USE_MYSQL') && USE_MYSQL && function_exists('friender_db') && is_int($userId)) {
  try {
    $pdo = friender_db();
    $pdo->beginTransaction();

    // 1. Configuration (friend request, messaging, scheduling, safety, settings)
    $friendRequest = isset($payload['friendRequest']) ? json_encode($payload['friendRequest'], JSON_UNESCAPED_SLASHES) : null;
    $messaging = isset($payload['messaging']) ? json_encode($payload['messaging'], JSON_UNESCAPED_SLASHES) : null;
    $scheduling = isset($payload['scheduling']) ? json_encode($payload['scheduling'], JSON_UNESCAPED_SLASHES) : null;
    $safety = isset($payload['safety']) ? json_encode($payload['safety'], JSON_UNESCAPED_SLASHES) : null;
    $settings = isset($payload['settings']) ? json_encode($payload['settings'], JSON_UNESCAPED_SLASHES) : null;
    $settingsInitialized = !empty($payload['settingsInitialized']) ? 1 : 0;
    $stmt = $pdo->prepare('INSERT INTO sync_config (user_id, friend_request, messaging, scheduling, safety, settings, settings_initialized, synced_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE friend_request = VALUES(friend_request), messaging = VALUES(messaging), scheduling = VALUES(scheduling), safety = VALUES(safety), settings = VALUES(settings), settings_initialized = VALUES(settings_initialized), synced_at = VALUES(synced_at)');
    $stmt->execute([$userId, $friendRequest, $messaging, $scheduling, $safety, $settings, $settingsInitialized, $syncedAt]);

    // 2. Message segments
    $pdo->prepare('DELETE FROM sync_segments WHERE user_id = ?')->execute([$userId]);
    if (!empty($payload['segments']) && is_array($payload['segments'])) {
      $stmt = $pdo->prepare('INSERT INTO sync_segments (user_id, segment_id, title, message, sort_order) VALUES (?, ?, ?, ?, ?)');
      foreach ($payload['segments'] as $i => $s) {
        $segId = is_array($s) ? ($s['id'] ?? $s['segment_id'] ?? (string)$i) : (string)$i;
        $title = is_array($s) ? ($s['title'] ?? $s['name'] ?? '') : '';
        $message = is_array($s) ? ($s['message'] ?? $s['content'] ?? '') : (string)$s;
        $stmt->execute([$userId, $segId, $title, $message, $i]);
      }
    }

    // 3. Message groups
    $pdo->prepare('DELETE FROM sync_groups WHERE user_id = ?')->execute([$userId]);
    if (!empty($payload['groups']) && is_array($payload['groups'])) {
      $stmt = $pdo->prepare('INSERT INTO sync_groups (user_id, group_id, title, items, sort_order) VALUES (?, ?, ?, ?, ?)');
      foreach ($payload['groups'] as $i => $g) {
        $gId = is_array($g) ? ($g['id'] ?? $g['group_id'] ?? (string)$i) : (string)$i;
        $title = is_array($g) ? ($g['title'] ?? $g['name'] ?? '') : '';
        $items = is_array($g) && isset($g['items']) ? json_encode($g['items'], JSON_UNESCAPED_SLASHES) : null;
        $stmt->execute([$userId, $gId, $title, $items, $i]);
      }
    }

    // 4. Automation state and counters
    $state = $payload['state'] ?? [];
    $stmt = $pdo->prepare('INSERT INTO sync_automation_state (user_id, status, friend_requests_sent, messages_sent, session_friend_requests, session_messages, errors, last_reset_date, session_start_time, synced_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE status = VALUES(status), friend_requests_sent = VALUES(friend_requests_sent), messages_sent = VALUES(messages_sent), session_friend_requests = VALUES(session_friend_requests), session_messages = VALUES(session_messages), errors = VALUES(errors), last_reset_date = VALUES(last_reset_date), session_start_time = VALUES(session_start_time), synced_at = VALUES(synced_at)');
    $stmt->execute([
      $userId,
      $state['status'] ?? 'stopped',
      (int)($state['friendRequestsSent'] ?? 0),
      (int)($state['messagesSent'] ?? 0),
      (int)($state['sessionFriendRequests'] ?? 0),
      (int)($state['sessionMessages'] ?? 0),
      (int)($state['errors'] ?? 0),
      $state['lastResetDate'] ?? null,
      isset($state['sessionStartTime']) ? (int)$state['sessionStartTime'] : null,
      $syncedAt
    ]);

    // 5. Activity logs
    $pdo->prepare('DELETE FROM sync_activity_log WHERE user_id = ?')->execute([$userId]);
    if (!empty($payload['activityLog']) && is_array($payload['activityLog'])) {
      $stmt = $pdo->prepare('INSERT INTO sync_activity_log (user_id, timestamp, type, message, data, sort_order) VALUES (?, ?, ?, ?, ?, ?)');
      foreach ($payload['activityLog'] as $i => $a) {
        $ts = is_array($a) ? ($a['timestamp'] ?? '') : '';
        $type = is_array($a) ? ($a['type'] ?? '') : '';
        $msg = is_array($a) ? ($a['message'] ?? '') : (string)$a;
        $data = is_array($a) && isset($a['data']) ? json_encode($a['data'], JSON_UNESCAPED_SLASHES) : null;
        $stmt->execute([$userId, $ts, $type, $msg, $data, $i]);
      }
    }

    // 6. Pending friend requests with message tracking
    $pdo->prepare('DELETE FROM sync_pending_friend_requests WHERE user_id = ?')->execute([$userId]);
    if (!empty($payload['pendingFriendRequests']) && is_array($payload['pendingFriendRequests'])) {
      $stmt = $pdo->prepare('INSERT INTO sync_pending_friend_requests (user_id, url, name, profile_id, sent_at, message_sent, message_sent_at, stored_message) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');
      foreach ($payload['pendingFriendRequests'] as $r) {
        if (!is_array($r)) continue;
        $stmt->execute([
          $userId,
          $r['url'] ?? null,
          $r['name'] ?? null,
          $r['profileId'] ?? null,
          (int)($r['sentAt'] ?? 0),
          !empty($r['messageSent']) ? 1 : 0,
          isset($r['messageSentAt']) ? (int)$r['messageSentAt'] : null,
          $r['storedMessage'] ?? null
        ]);
      }
    }

    // 7. Position / resume data
    $stmt = $pdo->prepare('INSERT INTO sync_position_resume (user_id, last_search_position, last_search_url, synced_at) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE last_search_position = VALUES(last_search_position), last_search_url = VALUES(last_search_url), synced_at = VALUES(synced_at)');
    $stmt->execute([$userId, (int)($payload['lastSearchPosition'] ?? 0), $payload['lastSearchUrl'] ?? null, $syncedAt]);

    // 8. Pending message retry (temporary message queues)
    $pdo->prepare('DELETE FROM sync_pending_message_retry WHERE user_id = ?')->execute([$userId]);
    if (!empty($payload['pendingMessageRetry']) && is_array($payload['pendingMessageRetry'])) {
      $stmt = $pdo->prepare('INSERT INTO sync_pending_message_retry (user_id, profile_id, profile_name, profile_url, trigger, stored_message, attempts, last_attempt) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');
      foreach ($payload['pendingMessageRetry'] as $r) {
        if (!is_array($r)) continue;
        $stmt->execute([
          $userId,
          $r['profileId'] ?? null,
          $r['profileName'] ?? null,
          $r['profileUrl'] ?? null,
          $r['trigger'] ?? null,
          $r['storedMessage'] ?? null,
          (int)($r['attempts'] ?? 0),
          isset($r['lastAttempt']) ? (int)$r['lastAttempt'] : null
        ]);
      }
    }

    // 9. Tracking sets (duplicate prevention)
    $trackingKeys = ['acceptedNotificationsMessageSent', 'incomingRequestsTracked', 'acceptedRequestsTracked', 'declinedRequestsTracked', 'acceptedRequestsMessageSent'];
    $stmt = $pdo->prepare('INSERT INTO sync_tracking_sets (user_id, set_key, set_value) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE set_value = VALUES(set_value)');
    foreach ($trackingKeys as $key) {
      if (array_key_exists($key, $payload)) {
        $val = is_array($payload[$key]) ? json_encode($payload[$key], JSON_UNESCAPED_SLASHES) : json_encode($payload[$key]);
        $stmt->execute([$userId, $key, $val]);
      }
    }

    // 10. Misc key-value (tab IDs, lastMessageSentTo, migrations)
    $miscKeys = ['lastPendingRetryRunAt', 'lastMessageSentTo', 'incomingTrackingMigration', 'incomingTrackingMigrationV3', 'incomingTrackingMigrationV4'];
    $stmt = $pdo->prepare('INSERT INTO sync_misc (user_id, misc_key, misc_value) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE misc_value = VALUES(misc_value)');
    foreach ($miscKeys as $key) {
      if (array_key_exists($key, $payload)) {
        $val = json_encode($payload[$key], JSON_UNESCAPED_SLASHES);
        $stmt->execute([$userId, $key, $val]);
      }
    }

    // 11. Full payload backup (for load)
    $json = json_encode($payload, JSON_UNESCAPED_SLASHES);
    $stmt = $pdo->prepare('INSERT INTO sync_data (user_id, payload, synced_at) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE payload = ?, synced_at = ?');
    $stmt->execute([$userId, $json, $syncedAt, $json, $syncedAt]);

    $pdo->commit();
    echo json_encode(['ok' => true, 'syncedAt' => $syncedAt]);
    exit;
  } catch (PDOException $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    error_log('ConnectWise save DB error: ' . $e->getMessage());
    echo json_encode(['ok' => false, 'error' => 'Failed to save']);
    exit;
  }
}

// File-based
$dataFile = STORAGE_DIR . '/sync/' . $userId . '.json';
$dataDir = dirname($dataFile);
if (!is_dir($dataDir)) mkdir($dataDir, 0755, true);
if (!file_put_contents($dataFile, json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES))) {
  echo json_encode(['ok' => false, 'error' => 'Failed to save']);
  exit;
}
echo json_encode(['ok' => true, 'syncedAt' => $payload['_syncedAt']]);
exit;

function resolveTokenToUserIdFile($token) {
  if (!strlen($token)) return null;
  $tokenDir = STORAGE_DIR . '/tokens';
  if (!is_dir($tokenDir)) return null;
  foreach (glob($tokenDir . '/*.txt') as $f) {
    $content = file_get_contents($f);
    $lines = explode("\n", $content);
    $stored = trim($lines[0] ?? '');
    $expires = (int)($lines[1] ?? 0);
    if ($stored === $token && $expires > time()) {
      return basename($f, '.txt');
    }
  }
  return null;
}
